<?php
class didar_api{

	/**
	 * Save error details to order meta (public wrapper)
	 */
	public static function log_error($order_id, $error_obj, $function_name = '', $source = 'Didar', $api_payload = null){
		self::save_error_details($order_id, $error_obj, $function_name, $source, $api_payload);
	}

	/**
	 * Save error details to order meta
	 */
	private static function save_error_details($order_id, $error_obj, $function_name = '', $source = 'Didar', $api_payload = null){
		if(!isset($error_obj->Error) && !isset($error_obj->Message))
			return;
		
		// Extract api_payload from error_obj if not provided (for consistent url/headers/body logging)
		if($api_payload === null && isset($error_obj->OperationInfo->ApiPayload) && is_array($error_obj->OperationInfo->ApiPayload)){
			$api_payload = $error_obj->OperationInfo->ApiPayload;
		}
		
		$error_msg = isset($error_obj->Error) ? $error_obj->Error : $error_obj->Message;
		$error_code = isset($error_obj->Code) ? $error_obj->Code : '';
		$operation_name = isset($error_obj->OperationInfo->OperationName) ? $error_obj->OperationInfo->OperationName : 'نامشخص';
		$endpoint = isset($error_obj->OperationInfo->Endpoint) ? $error_obj->OperationInfo->Endpoint : 'نامشخص';
		
		// Get function name from backtrace if not provided
		if(empty($function_name)){
			$backtrace = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 3);
			if(isset($backtrace[2]['function'])){
				$function_name = $backtrace[2]['function'];
				if(isset($backtrace[2]['class'])){
					$function_name = $backtrace[2]['class'] . '::' . $function_name;
				}
			}
		}
		
		// Get current date/time
		$error_datetime = current_time('mysql');
		$error_timestamp = current_time('timestamp');
		
		// Save all error details
		update_post_meta($order_id, 'didar_error_source', $source);
		update_post_meta($order_id, 'didar_error_message', $error_msg);
		update_post_meta($order_id, 'didar_error_code', $error_code);
		update_post_meta($order_id, 'didar_error_operation', $operation_name);
		update_post_meta($order_id, 'didar_error_endpoint', $endpoint);
		update_post_meta($order_id, 'didar_error_function', $function_name);
		update_post_meta($order_id, 'didar_error_datetime', $error_datetime);
		update_post_meta($order_id, 'didar_error_timestamp', $error_timestamp);
		
		// Save detailed error text for backward compatibility
		$detailed_error = "منبع: $source | عملیات: $operation_name | Endpoint: $endpoint | تابع: $function_name | خطا: $error_msg";
		if($error_code){
			$detailed_error .= " | کد خطا: $error_code";
		}
		$detailed_error .= " | تاریخ/زمان: $error_datetime";
		update_post_meta($order_id, 'didar_error_detail', $detailed_error);
		
		// Save simple message for backward compatibility
		update_post_meta($order_id, 'didar_msg', $error_msg);
		
		// Save API payload (url, headers, body) - required for all API error logs
		if($api_payload !== null && is_array($api_payload)){
			update_post_meta($order_id, 'didar_error_api_url', isset($api_payload['url']) ? $api_payload['url'] : '');
			update_post_meta($order_id, 'didar_error_api_headers', isset($api_payload['headers']) ? json_encode($api_payload['headers'], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT) : '');
			update_post_meta($order_id, 'didar_error_api_body', isset($api_payload['body']) ? $api_payload['body'] : '');
		}
	}

	/**
	 * Build ApiPayload array (url, headers, body) for error logging
	 */
	private static function build_api_payload($url, $headers, $body){
		return [
			'url' => $url,
			'headers' => is_array($headers) ? $headers : (array)$headers,
			'body' => is_string($body) ? $body : json_encode($body, JSON_UNESCAPED_UNICODE)
		];
	}

	private static function send_request($router,$params=[],$operation_name=''){

		$opt = get_option( 'did_option', [] );
		$url = "https://app.didar.me/api/$router?apikey={$opt['didar_api']}";
		$headers = array('Content-Type:application/json');
		$body = json_encode($params, JSON_UNESCAPED_UNICODE);
		$api_payload = self::build_api_payload($url, $headers, $body);
		
		$ch  = curl_init($url);
		curl_setopt( $ch, CURLOPT_POST, 1);
		curl_setopt( $ch, CURLOPT_FOLLOWLOCATION, 0);
		curl_setopt( $ch, CURLOPT_SSL_VERIFYPEER, 0);
		curl_setopt( $ch, CURLOPT_SSL_VERIFYHOST, 0);
		curl_setopt( $ch, CURLOPT_POSTFIELDS, $body);
		curl_setopt( $ch, CURLOPT_HTTPHEADER, $headers);
		curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true );
		$resp = curl_exec($ch);
		$curl_error = curl_error($ch);
		$http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		curl_close($ch);
		
		$result = ($resp !== false) ? json_decode($resp) : null;
		
		// Determine if we have an error (API error, curl failure, invalid JSON, or HTTP error)
		$has_error = false;
		$error_msg = '';
		$error_code = '';
		
		if($resp === false || $curl_error){
			$has_error = true;
			$error_msg = $curl_error ?: 'خطای اتصال به سرور';
			$error_code = 'CURL';
			$result = (object)['Error' => $error_msg, 'Code' => $error_code];
		} elseif($result === null){
			$has_error = true;
			$error_msg = 'پاسخ نامعتبر از API (JSON نامعتبر)';
			$error_code = 'JSON';
			$result = (object)['Error' => $error_msg, 'Code' => $error_code];
		} elseif(isset($result->Error) || isset($result->Message)){
			$has_error = true;
			$error_msg = isset($result->Error) ? $result->Error : $result->Message;
			$error_code = isset($result->Code) ? $result->Code : '';
		} elseif($http_code >= 400){
			$has_error = true;
			$error_msg = "خطای HTTP: $http_code";
			$error_code = (string)$http_code;
			if(!isset($result->Error) && !isset($result->Message)){
				$result->Error = $error_msg;
				$result->Code = $error_code;
			}
		}
		
		if($has_error){
			$log_message = "خطا در API دیدار";
			if($operation_name){
				$log_message .= " - عملیات: $operation_name";
			}
			$log_message .= " - Endpoint: $router";
			$log_message .= " - خطا: $error_msg";
			if($error_code){
				$log_message .= " - کد خطا: $error_code";
			}
			$log_message .= " | URL: $url | Body: " . substr($body, 0, 200) . (strlen($body) > 200 ? '...' : '');
			error_log($log_message);
			
			if(!isset($result->OperationInfo)){
				$result->OperationInfo = (object)[];
			}
			$result->OperationInfo->OperationName = $operation_name;
			$result->OperationInfo->Endpoint = $router;
			$result->OperationInfo->ApiPayload = $api_payload;
		}
		
		return $result;
	}


	/**
	 * Append api_payload (url, headers, body) to log message for PHP error_log
	 */
	private static function format_api_payload_for_log($api_payload){
		if(empty($api_payload) || !is_array($api_payload)){
			return '';
		}
		$parts = [];
		if(!empty($api_payload['url'])){
			$parts[] = "URL: " . $api_payload['url'];
		}
		if(!empty($api_payload['headers'])){
			$headers_str = is_array($api_payload['headers']) ? json_encode($api_payload['headers'], JSON_UNESCAPED_UNICODE) : $api_payload['headers'];
			$parts[] = "Headers: " . $headers_str;
		}
		if(isset($api_payload['body'])){
			$parts[] = "Body: " . $api_payload['body'];
		}
		return empty($parts) ? '' : " | " . implode(' | ', $parts);
	}

	public static function get_kariz_list(){
		$result = self::send_request('pipeline/list/0',[],'دریافت لیست کاریزها');
		$api_payload = isset($result->OperationInfo->ApiPayload) ? $result->OperationInfo->ApiPayload : null;
		$payload_log = self::format_api_payload_for_log($api_payload);
		
		// Check for errors
		if(isset($result->Error) || isset($result->Message)){
			$error_msg = isset($result->Error) ? $result->Error : $result->Message;
			$error_code = isset($result->Code) ? $result->Code : '';
			$log_message = "خطا در دریافت لیست کاریزها از API دیدار";
			$log_message .= " - Endpoint: pipeline/list/0";
			$log_message .= " - خطا: $error_msg";
			if($error_code){
				$log_message .= " - کد خطا: $error_code";
			}
			$log_message .= $payload_log;
			error_log($log_message);
			return [];
		}
		
		// Check if Response exists and is valid
		if(!isset($result->Response) || !is_array($result->Response)){
			$log_message = "خطا در دریافت لیست کاریزها: پاسخ نامعتبر از API";
			$log_message .= " - Endpoint: pipeline/list/0";
			$log_message .= " - Response: " . json_encode($result, JSON_UNESCAPED_UNICODE);
			$log_message .= $payload_log;
			error_log($log_message);
			return [];
		}
		
		return $result->Response;
	}

	public static function get_user_list(){
		return self::send_request('User/List',[])->Response;
	}


	private static function get_price($product_id){
		if($price = get_post_meta($product_id,'_sale_price',true))
			return $price;
		if($price = get_post_meta($product_id,'_regular_price',true))
			return $price;
		if($price = get_post_meta($product_id,'_price',true))
			return $price;
	}

	public static function has_user($mobile){

		$user = self::send_request('contact/getbyphonenumber',['MobilePhone'=>$mobile])->Response;
		return empty($user)?false:$user[0];
		/*$user = self::send_request('contact/personsearch',['Criteria'=>['LeadType'=>3,'Keywords'=>$user_name]])->Response;
        if($user->TotalCount==0)
        return false;
        else
        return $user->List[0];*/
	}


	public static function create_user($args=[]){
		if($user = self::has_user($args['MobilePhone']))
			return $user;
		$resp = self::send_request('contact/save',['Contact'=>$args],'ایجاد شخص/مخاطب');
		if(isset($resp->Error) || isset($resp->Message)){
			$api_payload = isset($resp->OperationInfo->ApiPayload) ? $resp->OperationInfo->ApiPayload : null;
			$resp->OperationInfo = (object)['OperationName' => 'ایجاد شخص/مخاطب', 'Endpoint' => 'contact/save'];
			if($api_payload){
				$resp->OperationInfo->ApiPayload = $api_payload;
			}
		}
        return isset($resp->Response)?$resp->Response:$resp;
	}

	public static function has_product($pid){
		return self::send_request('product/search',['Criteria'=>['Query'=>urlencode($pid)]]);
	}

	/**
	 * Check if a SKU exists in WooCommerce
	 */
	private static function sku_exists_in_woocommerce($sku){
		global $wpdb;
		if(empty($sku)){
			return false;
		}
		$exists = $wpdb->get_var($wpdb->prepare(
			"SELECT COUNT(*) FROM {$wpdb->postmeta} 
			WHERE meta_key = '_sku' AND meta_value = %s",
			$sku
		));
		return $exists > 0;
	}

	/**
	 * Check if a SKU exists in Didar
	 */
	private static function sku_exists_in_didar($sku){
		if(empty($sku)){
			return false;
		}
		$result = self::send_request('product/getproductbycodes', ['Code' => [$sku]], 'بررسی وجود SKU در دیدار');
		// Only return true if we get a valid response with products
		// If there's an error or no products, SKU doesn't exist
		if(isset($result->Response) && isset($result->Response->Products) && is_array($result->Response->Products) && !empty($result->Response->Products)){
			return true;
		}
		return false;
	}
	

	/**
	 * Generate a unique SKU that doesn't exist in WooCommerce or Didar
	 */
	private static function generate_unique_sku(){
		$max_attempts = 100;
		$attempt = 0;
		
		while($attempt < $max_attempts){
			// Generate random SKU: DIDAR- followed by random alphanumeric string
			$random_string = 'DIDAR-' . strtoupper(wp_generate_password(12, false, false));
			
			// Check if SKU exists in WooCommerce
			if(self::sku_exists_in_woocommerce($random_string)){
				$attempt++;
				continue;
			}
			
			// Check if SKU exists in Didar
			if(self::sku_exists_in_didar($random_string)){
				$attempt++;
				continue;
			}
			
			// SKU is unique
			return $random_string;
		}
		
		// Fallback: use timestamp-based SKU if random generation fails
		return 'DIDAR-' . strtoupper(wp_generate_password(8, false, false)) . '-' . time();
	}

	public static function get_product_by_code($id){
		global $wpdb;
		$sku = get_post_meta($id,'_sku',true);
		//$parent = $wpdb->get_var("select post_parent from $wpdb->posts join $wpdb->postmeta on ID=post_id and meta_key='_sku' and meta_value='$sku'");
		$parent = $wpdb->get_var("select post_parent from $wpdb->posts where ID=$id");
		$idx     = empty($parent)?$sku:get_post_meta($parent,'_sku',true);
		$resp = self::send_request('product/getproductbycodes',['Code'=>[$idx]],'جستجوی محصول');
		if(isset($resp->Error) || isset($resp->Message)){
			$api_payload = isset($resp->OperationInfo->ApiPayload) ? $resp->OperationInfo->ApiPayload : null;
			$resp->OperationInfo = (object)['OperationName' => 'جستجوی محصول', 'Endpoint' => 'product/getproductbycodes'];
			if($api_payload){
				$resp->OperationInfo->ApiPayload = $api_payload;
			}
			return $resp;
		}
		$product = $resp->Response->Products[0];

		if(empty($product))
			return;
		if(empty($parent))
			return $product;
		foreach($product->Variants as $Variants){
			if($Variants->VariantCode == $sku)
				return $Variants;
		}
		//return self::send_request('product/getproductbycodes',['Code'=>[$idx]])->Response->Products;
	}

	public static function create_product($item){

		global $wpdb;
		$product_id = $item->get_product_id();
		$variation_id = $item->get_variation_id();
		
		// Step 1: Check and generate SKU for main product if needed
		$product_sku = did_get_sku($product_id);
		if(empty($product_sku)){
			$product_sku = self::generate_unique_sku();
			update_post_meta($product_id, '_sku', $product_sku);
		}
		
		// Step 2: If variant is used, check and generate SKU for variant if needed
		if(!empty($variation_id)){
			$variant_sku = did_get_sku($variation_id);
			if(empty($variant_sku)){
				$variant_sku = self::generate_unique_sku();
				update_post_meta($variation_id, '_sku', $variant_sku);
			}
		}
		
		// Step 3: Now search for product in Didar (use variation ID if exists, otherwise product ID)
		$id = empty($variation_id) ? $product_id : $variation_id;
		$product = self::get_product_by_code($id);
		if(!empty($product)){
			update_post_meta( $id, 'didar_id', $product->Id );
			return empty($product->ProductId)?[$product->Id,null]:[$product->ProductId,$product->Id];
		}

		/*if($product = get_post_meta( $item->get_product_id(), 'didar_id', true )){
			if($variant = get_post_meta( $item->get_variation_id(), 'didar_id', true ))
				return [$product,$variant];
			else if(empty($item->get_variation_id()))
				return [$product,null];
		}*/


		$opt = get_option( 'did_option', [] );
		$product_category_id = isset($opt['product_category']) && !empty($opt['product_category']) ? $opt['product_category'] : '';
		
		$title = get_the_title($product_id);
		
		// Use product SKU from above (already checked and generated if needed)
		// $product_sku is already set in Step 1
		
		$args  = [
			'Title'             => $title,
			'Code'              => $product_sku,
			'Unit'              => 'IRR',
			'Description'       => '',
			'TitleForInvoice'   => $title,
			'ProductCategoryId' => $product_category_id,
		];
		if(empty($variation_id)){
			$args['UnitPrice']=$item->get_subtotal();
			// Check currency from plugin settings, fallback to WooCommerce settings
			$opt_currency = get_option('did_option', []);
			$currency_type = isset($opt_currency['currency_type']) ? $opt_currency['currency_type'] : get_option('woocommerce_currency', 'IRR');
			if($currency_type == 'IRT')
				$args['UnitPrice']*= 10;
		}


		if($didar = get_post_meta( $product_id, 'didar_id', true )){
			$args['Id'] = $didar;
		}

		$variations = $wpdb->get_results("select * from $wpdb->posts where post_type='product_variation' and post_status='publish' and post_parent=".$product_id);
		if(!empty($variations)){
			$args['Variants'] = [];
			foreach($variations as $i=>$variation){
				$price = self::get_price($variation->ID);
				// Check currency from plugin settings, fallback to WooCommerce settings
				$opt_currency = get_option('did_option', []);
				$currency_type = isset($opt_currency['currency_type']) ? $opt_currency['currency_type'] : get_option('woocommerce_currency', 'IRR');
				if($currency_type == 'IRT')
					$price*= 10;

				// Get variant SKU
				$variant_sku = did_get_sku($variation->ID);
				
				// If variant SKU is empty, generate unique SKU and update variant
				if(empty($variant_sku)){
					$variant_sku = self::generate_unique_sku();
					update_post_meta($variation->ID, '_sku', $variant_sku);
				}

				$args['Variants'][$i] = [
					'UnitPrice'       => $price,
					'Title'           => $variation->post_title,
					'VariantCode'     => $variant_sku,
					'TitleForInvoice' => $variation->post_title,
					'IsDefault'       => false
				];
				if($didid = get_post_meta( $variation->ID, 'didar_id', true )){
					$args['Variants'][$i]['Id']        = $didid;
					$args['Variants'][$i]['ProductId'] = $didar;
				}
			}
		}

		$had_update_attempt = isset($args['Id']);
		$product = self::send_request('product/save',['Product'=>$args],'ایجاد/به‌روزرسانی محصول');

		// If "product not found" and we were trying to update - product was deleted in Didar
		// Clear invalid didar_ids and retry as CREATE (new product)
		if(!isset($product->Response)){
			$error_msg = isset($product->Error) ? $product->Error : (isset($product->Message) ? $product->Message : '');
			$is_product_not_found = (stripos($error_msg, 'product not found') !== false);
			
			if($had_update_attempt && $is_product_not_found){
				// Clear invalid didar_id from main product
				delete_post_meta($product_id, 'didar_id');
				// Clear didar_id from all variations
				if(!empty($variations)){
					foreach($variations as $v){
						delete_post_meta($v->ID, 'didar_id');
					}
					// Rebuild args without Id and without variant Ids/ProductIds for CREATE
					unset($args['Id']);
					foreach($args['Variants'] as $i => $variant_data){
						unset($args['Variants'][$i]['Id']);
						unset($args['Variants'][$i]['ProductId']);
					}
				} else {
					unset($args['Id']);
				}
				// Retry as create (new product)
				$product = self::send_request('product/save',['Product'=>$args],'ایجاد/به‌روزرسانی محصول');
			}
		}
		
		if(!isset($product->Response)){
			if(isset($product->Error) || isset($product->Message)){
				$api_payload = isset($product->OperationInfo->ApiPayload) ? $product->OperationInfo->ApiPayload : null;
				$product->OperationInfo = (object)['OperationName' => 'ایجاد/به‌روزرسانی محصول', 'Endpoint' => 'product/save'];
				if($api_payload){
					$product->OperationInfo->ApiPayload = $api_payload;
				}
			}
			return $product;
		}
		$product = $product->Response;

		if(isset($product->Id))
			update_post_meta($product_id, 'didar_id',$product->Id);
		if(!empty($product->Variants)){
			foreach($product->Variants as $variant){
				update_post_meta(did_get_variation_id_by_sku($variant->VariantCode), 'didar_id',$variant->Id);
				//update_post_meta($variant->VariantCode, 'didar_id',$variant->Id);
			}
		}
		$var_id = empty($variation_id)?null:get_post_meta( $variation_id, 'didar_id', true );
		return [get_post_meta( $product_id, 'didar_id', true ),$var_id];
	}

	public static function get_category(){
		return self::send_request('product/categories',[])->Response;
	}

	public static function create_order($args=[]){

		return self::send_request('deal/save',$args,'ایجاد معامله');
	}

	public static function save_order($order_id){
		
        $opt   = get_option( 'did_option', [] );
		$order = new WC_Order($order_id);
		
		// Validate email before adding to args
		$billing_email = $order->get_billing_email();
		$valid_email = false;
		if(!empty($billing_email) && is_string($billing_email)){
			// Basic email validation using regex (compatible with all PHP versions)
			// Pattern: local@domain.tld
			// Local part: alphanumeric, dots, hyphens, underscores
			// Domain: alphanumeric, dots, hyphens
			// TLD: at least 2 letters
			$email_pattern = '/^[a-zA-Z0-9._-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/';
			$valid_email = preg_match($email_pattern, trim($billing_email)) === 1;
		}
		
		$args = [
			'Type'=>'Person',
			'FirstName'      => $order->get_billing_first_name(),
			'LastName'       => $order->get_billing_last_name(),
			'DisplayName'    => $order->get_formatted_billing_full_name(),
			'MobilePhone'    => $order->get_billing_phone(),
			'IsCustomer'     => true,
			//'CustomerCode' => $order->get_customer_id(),
			'ZipCode'        => $order->get_billing_postcode(),
			'CityTitle'      => $order->get_billing_city(),
			'ProvinceTitle'  => $order->get_billing_state(),
			'CompanyName'    => $order->get_billing_company(),
			'VisibilityType' => 'All',
			'Addresses'      => ['KeyValues'=>[
				[
					'Key'   => "آدرس",
					'Value' => preg_replace("/<[^>]+>/is"," ",$order->get_formatted_billing_address()),
				],
			]],
		];
		
		// Add Email only if it's valid
		if($valid_email){
			$args['Email'] = $billing_email;
		}
		$user = self::create_user($args);
        if(!isset($user->Id)){
			// Check if it's an error from create_user
			if(isset($user->Error) || isset($user->Message)){
				$api_payload = isset($user->OperationInfo->ApiPayload) ? $user->OperationInfo->ApiPayload : null;
				$user->OperationInfo = (object)['OperationName' => 'ایجاد شخص/مخاطب', 'Endpoint' => 'contact/save'];
				if($api_payload){
					$user->OperationInfo->ApiPayload = $api_payload;
				}
				self::save_error_details($order_id, $user, 'didar_api::save_order', 'Didar', $api_payload);
			}
            return $user;
		}

		$product = [];
		$items   = $order->get_items();
		foreach ( $items as $item ) {
			$did_prod = self::create_product($item);
            if(is_object($did_prod)){
				// Check if it's an error from create_product
				if(isset($did_prod->Error) || isset($did_prod->Message)){
					$api_payload = isset($did_prod->OperationInfo->ApiPayload) ? $did_prod->OperationInfo->ApiPayload : null;
					$did_prod->OperationInfo = (object)['OperationName' => 'ایجاد/به‌روزرسانی محصول', 'Endpoint' => 'product/save'];
					if($api_payload){
						$did_prod->OperationInfo->ApiPayload = $api_payload;
					}
					self::save_error_details($order_id, $did_prod, 'didar_api::save_order', 'Didar', $api_payload);
				}
                return $did_prod;
			}
			

			$price = intval($item->get_subtotal()/$item->get_quantity());
			// Check currency from plugin settings, fallback to WooCommerce settings
			$currency_type = isset($opt['currency_type']) ? $opt['currency_type'] : get_option('woocommerce_currency', 'IRR');
			if($currency_type == 'IRT')
				$price*= 10;

			$product_item = [
				'ProductId'   => $did_prod[0],
				'Quantity'    => $item->get_quantity(),
				'UnitPrice'   => $price,
				'Discount'    => $item->get_subtotal_tax(),
				'ProductCode' => (empty($item->get_variation_id())?did_get_sku($item->get_product_id()):did_get_sku($item->get_variation_id()) ),
				//'ProductCode' => (empty($item->get_variation_id())?$item->get_product_id():$item->get_variation_id()),
			];
			
			// Only add VariantId if item has variation_id (product has variant)
			// and did_prod[1] has a valid non-null value
			if(!empty($item->get_variation_id()) && isset($did_prod[1]) && $did_prod[1] !== null && $did_prod[1] !== '' && $did_prod[1] !== false){
				$product_item['VariantId'] = $did_prod[1];
			}
			
			$product[] = $product_item;
		}
		$tax  = empty($order->get_total_tax())?0:($order->get_total_tax()/$order->get_total()*100);
		
		// Get deal type from configuration
		$order_status_key = 'wc-' . $order->get_status();
		$status_config = isset($opt['status_config']) ? $opt['status_config'] : [];
		
		// Get deal type from status_config, fallback to old logic for backward compatibility
		if(isset($status_config[$order_status_key])){
			$status = intval($status_config[$order_status_key]);
		} else {
			// Backward compatibility: use old switch logic if status_config not set
			$status = 0;
			switch($order->get_status()){
				case 'completed':$status=1;break;
				case 'cancelled':
				case 'refunded':
				case 'failed':$status=2;break;
				default:$status=0;break;
			}
		}
		// Check currency from plugin settings, fallback to WooCommerce settings
		$currency_type = isset($opt['currency_type']) ? $opt['currency_type'] : get_option('woocommerce_currency', 'IRR');
		if($currency_type == 'IRT'){
			$price= $order->get_total()*10;
			$tax *= 10;
		}else
			$price=$order->get_total();
		$soid = (isset($opt['soid']) && $opt['soid']=='on') ? ' '.$order->get_order_number() : '';
		
		// Get order registration date
		// Use timestamp to ensure we always get Gregorian (miladi) date, not Persian (shamsi)
		// This is critical because Persian calendar plugins may convert dates to shamsi format
		// which causes "SqlDateTime overflow" error in the API
		$order_date = $order->get_date_created();
		$order_date_formatted = null;
		if($order_date){
			// Get timestamp (Unix timestamp is always Gregorian, not affected by calendar plugins)
			$timestamp = $order_date->getTimestamp();
			if($timestamp && $timestamp > 0){
				// Format as ISO 8601 in UTC timezone using gmdate()
				// gmdate() always uses Gregorian calendar regardless of WordPress settings
				$order_date_formatted = gmdate('Y-m-d\TH:i:s', $timestamp);
			}
		}
		
		// Get owner ID - if multiple users selected, choose randomly
		$owner_id = '';
		if(isset($opt['user'])){
			$users = $opt['user'];
			if(!is_array($users)){
				$users = !empty($users) ? [$users] : [];
			}
			if(!empty($users)){
				// If multiple users, select randomly
				if(count($users) > 1){
					$owner_id = $users[array_rand($users)];
				} else {
					$owner_id = $users[0];
				}
			}
		}
		
		$args = [
			'PersonId'        => $user->Id,
			'ContactId'       => $user->Id,
			'Status'          => $status,
			'Title'           => 'معامله '.$order->get_formatted_billing_full_name().$soid,
			'Price'           => $price,
			'Code'            => $order_id,
			//'RegisterDate'  => $order->get_date_created(),
			'IsPaid'          => true,
			'IsWon'           => ($status==1?true:false),
			'OwnerId'         => $owner_id,
			'PipelineStageId' => (isset($opt['kariz'])?$opt['kariz']:''),
		];
		
		// Add RegisterTime with order registration date in all cases
		if($order_date_formatted){
			$args['RegisterTime'] = $order_date_formatted;
		}
		
		// Add ChangeToWonTime for successful deals (status == 1)
		if($status == 1 && $order_date_formatted){
			$args['ChangeToWonTime'] = $order_date_formatted;
		}
		
		// Add ChangeToLossTime for unsuccessful deals (status == 2)
		if($status == 2 && $order_date_formatted){
			$args['ChangeToLossTime'] = $order_date_formatted;
		}
		
		//echo(json_encode(['Deal'=>$args,'DealItems'=>$product]));die;
		$out = self::create_order(['Deal'=>$args,'DealItems'=>$product]);
		
		// Check for errors and log them with operation info
		if(isset($out->Error) || isset($out->Message)){
			$api_payload = isset($out->OperationInfo->ApiPayload) ? $out->OperationInfo->ApiPayload : null;
			self::save_error_details($order_id, $out, 'didar_api::save_order', 'Didar', $api_payload);
		}
		
		//var_dump(['Deal'=>$args,'DealItems'=>$product]);
		//var_dump($out);
		return isset($out->Response)?$out->Response:$out;
	}
}